<?php
// *********************************************************************************************************************************
//
// RetrieveFXDataAPI.class.php
//
// *********************************************************************************************************************************
//
// #### Part of FX.php #####################################################
// #                                                                       #
// #  License: Artistic License (included with release)                    #
// # Web Site: www.iviking.org                                             #
// #                                                                       #
// #########################################################################
//
// Copyright (c) 2017 - 2019 Mark DeNyse
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//
// *********************************************************************************************************************************

require_once('RetrieveFXData.class.php');

require_once('RetrieveFXDataAPI/fmPDA/vLatest/fmDataAPI.class.php'); // Always use the latest version of fmDataAPI

class RetrieveFXDataAPI extends RetrieveFXData {
   protected $action;
   protected $fm;
   protected $logoutAfterCall;
   protected $sendData;
   protected $sendOptions;

   // *********************************************************************************************************************************
   function __construct(&$FX)
   {

      $this->fm               = NULL;
      $this->logoutAfterCall  = false;
      $this->sendData         = array();
      $this->sendOptions      = array();

      parent::__construct($FX);

      fmLogger('FX '. FX_VERSION . ' (Data API edition)');

      return;
    }

   // *********************************************************************************************************************************
   function cleanUp()
   {
      if ($this->logoutAfterCall) {
         $this->fm->apiLogout();
      }
      return;
   }

   // *********************************************************************************************************************************
   // Convert XML find operators into Data API equivalents
   function GetFindValue($operator, $value)
   {
      switch ($operator) {
         case 'eq':                             break;
         case 'cn':  $value = '*'. $value .'*'; break;
         case 'bw':  $value = $value .'*';      break;
         case 'ew':  $value = '*'. $value;      break;
         case 'gt':  $value = '>'. $value;      break;
         case 'gte': $value = '>='. $value;     break;
         case 'lt':  $value = '<'. $value;      break;
         case 'lte': $value = '<='. $value;     break;

         default:                               break;
      }

      return $value;
   }

   // *********************************************************************************************************************************
   // CreatePayload
   //
   // Prepare the data to send along with any options
   //
   function CreatePayload($action)
   {
      $this->sendData = array();
      $this->sendOptions = array();

      $requests = array();
      $query = array();

      foreach ($this->FX->dataParams as $param) {

         if ($action == $this->FX->actionArray['perform_find']) {

            if (is_array($param)) {
               // Walk the dataParams array. If we find an OR, create a new request to form a compound find.

               if ($param['name'] != '-lop') {
                  $query[$param['name']] = $this->GetFindValue($param['op'], $param['value']);
               }
               else if (count($query) > 0) {
                  $requests[] = $query;
                  $query = array();
               }
            }

         }

         else if (is_array($param)) {
            if (substr($param['name'], 0, 1) != '-') {                        // Ignore any parameters
               $this->sendData[$param['name']] = $this->GetFindValue($param['op'], $param['value']);
            }
         }

         else if ($param['name'] == '-script') {
            $this->sendOptions['script'] = $param['value'];
         }
         else if ($param['name'] == '-script.param') {
            $this->sendOptions['scriptParams'] = $param['value'];
         }
         else if ($param['name'] == '-script.prefind') {
            $this->sendOptions['scriptPrerequest'] = $param['value'];
         }
         else if ($param['name'] == '-script.prefind.param') {
            $this->sendOptions['scriptPrerequestParams'] = $param['value'];
         }
         else if ($param['name'] == '-script.presort') {
            $this->sendOptions['scriptPresort'] = $param['value'];
         }
         else if ($param['name'] == '-script.presort.param') {
            $this->sendOptions['scriptPresortParams'] = $param['value'];
         }
      }

      if ($action == $this->FX->actionArray['perform_find']) {
         if (count($query) > 0) {
            $requests[] = $query;
         }
         $this->sendOptions['query'] = $requests;
      }

      if (($action == $this->FX->actionArray['perform_find']) || ($action == $this->FX->actionArray['show_all'])) {
         if ($this->FX->groupSize != null) {
            $this->sendOptions['limit'] = $this->FX->groupSize;
         }
         if ($this->FX->currentSkip != 0) {
            $this->sendOptions['offset'] = $this->FX->currentSkip;
         }
     }

      foreach ($this->FX->sortParams as $sortParam) {
         $this->sendOptions['sort'][] = array('fieldName' => $sortParam['field'], 'sortOrder' => $sortParam['sortOrder']);
      }

      if ($this->FX->responseLayout != '') {
         $this->sendOptions['layoutResponse'] = $this->FX->responseLayout;
      }

      if (count($this->sendData) == 0) {
         $this->sendData = '';
      }

      return;
   }

   // *********************************************************************************************************************************
   // TranslateResponse
   //
   // Translate the response from the Data API into structures that FX understands.
   // The meta data can be optionally be retrieved.
   //
   function TranslateResponse($apiResult, $getMetaData = false, $metaDataRecordID = '')
   {
      $result = true;

      if (! $this->fm->getIsError($apiResult)) {

         $responseData = $this->fm->getResponseData($apiResult);

         foreach ($responseData as $record) {
            $fxData = array();

            // Map the fields in the record
            foreach ($record[FM_FIELD_DATA] as $fieldName => $value) {
               if (substr($fieldName, -1, 1) == ')') {                                                  // Repeating field?
                  $pieces = explode('(', $fieldName);
                  $fieldName = $pieces[0];
                  $fxData[$fieldName][] = $value;
               }
               else {
                  $fxData[$fieldName] = array($value);
               }
            }

            // Map any portal fields
            // We do not copy the recordID and modID fields even though the Data API provides them.
            // There would be no 'harm' in providing them.
            if (array_key_exists(FM_PORTAL_DATA, $record)) {
               foreach ($record[FM_PORTAL_DATA] as $portal) {
                  foreach ($portal as $portalRecord) {
                     foreach ($portalRecord as $portalFieldName => $portalFieldValue) {
                        if (($portalFieldName != FM_RECORD_ID) && ($portalFieldName != FM_MOD_ID)) {
                           if (substr($portalFieldName, -1, 1) == ')') {                               // Repeating field?
                              $pieces = explode('(', $portalFieldName);
                              $portalFieldName = $pieces[0];
                              $fxData[$portalFieldName][] = $portalFieldValue;
                           }
                           else {
                              $fxData[$portalFieldName][] = $portalFieldValue;
                           }
                        }
                     }
                  }
               }
            }

            $this->FX->currentData[$record[FM_RECORD_ID] .'.'. $record[FM_MOD_ID]] = $fxData;
         }
         $this->FX->foundCount = count($this->FX->currentData);

         if ($getMetaData) {                                            // Get the meta data as well?
            $result = $this->FMMetaData($metaDataRecordID);
            if (! FX::isError($result)) {
               $this->FX->fxError = 0;
            }
         }
         else {
            $this->FX->fxError = 0;
         }
      }
      else {
         $errorInfo = $this->fm->getMessageInfo($apiResult);

         $theError = $errorInfo[0];
         $result = new FX_Error($theError[FM_MESSAGE], $theError[FM_CODE]);
         $this->FX->fxError = $theError[FM_CODE];

         fmLogger('Found error(s):');
         fmLogger($errorInfo);
      }

      return $result;
   }

   // *********************************************************************************************************************************
   function GetRequestRecordID()
   {
      if (isset($this->FX->dataParams) && is_array($this->FX->dataParams) && count($this->FX->dataParams) > 0) {
         foreach ($this->FX->dataParams as $field) {
            if ($field['name'] == '-recid') {
               return $field['value'];
            }
         }
      }
      return '';
   }

   // *********************************************************************************************************************************
   function doQuery ($action)
   {
      $result = true;

      $this->action = $action;

      if ($this->fm == NULL) {
         $this->fm = new fmDataAPI($this->FX->database,
                                   $this->FX->dataServer . $this->FX->dataPortSuffix,
                                   $this->FX->DBUser, $this->FX->DBPassword);
      }

      switch ($this->action) {
         case $this->FX->actionArray['new']: {
            $result = $this->FMNew();
            break;
         }
         case $this->FX->actionArray['update']: {
            $result = $this->FMEdit();
            break;
         }
         case $this->FX->actionArray['delete']: {
            $result = $this->FMDelete();
            break;
         }
         case $this->FX->actionArray['duplicate']: {
            $result = $this->FMDuplicate();
            break;
         }
         case $this->FX->actionArray['perform_find']: {
            $result = $this->FMFind();
            break;
         }
         case $this->FX->actionArray['show_all']: {
            $result = $this->FMFindAll();
            break;
         }
         case $this->FX->actionArray['show_any']: {               // Use FindAll and just return the first record
            $result = $this->FMFindAll(1);
            break;
         }
         case $this->FX->actionArray['view_layout_objects']: {
            $result = $this->FMMetaData();
            break;
         }
         case $this->FX->actionArray['view_database_names']: {
            $result = $this->FMDBNames();
            break;
         }
         case $this->FX->actionArray['view_layout_names']: {
            $result = $this->FMLayoutNames();
            break;
         }
         case $this->FX->actionArray['view_script_names']: {
            $result = $this->FMScriptNames();
            break;
         }

         default: {
            $result = new FX_Error('Unknown action ='. $action);
            break;
         }
      }

      return $result;
   }


   // *********************************************************************************************************************************
   function FMNew()
   {
      $result = false;

      $this->CreatePayload($this->action);

      // GetKeyPairDataParams() will strip out duplicate key values - this is fine for everything but Find queries.
      $apiResult = $this->fm->apiCreateRecord($this->FX->layout, $this->sendData, $this->sendOptions);

      if (! $this->fm->getIsError($apiResult)) {
         $response = $this->fm->getResponse($apiResult);
         $recordID = $response[FM_RECORD_ID];

         // FX with the XML interface returns the record data and the meta data.
         // We need to make an additional calls to get the new record and the meta data.
         $result = $this->TranslateResponse($this->fm->apiGetRecord($this->FX->layout, $recordID), true, $recordID);
      }
      else {
         $errorInfo = $this->fm->getMessageInfo($apiResult);

         $theError = $errorInfo[0];
         $result = new FX_Error($theError[FM_MESSAGE], $theError[FM_CODE]);
         $this->FX->fxError = $theError[FM_CODE];

         fmLogger('Found error(s):');
         fmLogger($errorInfo);
      }

      return $result;
   }

   // *********************************************************************************************************************************
   function FMEdit()
   {
      $result = false;

      $this->CreatePayload($this->action);

      $apiResult = $this->fm->apiEditRecord($this->FX->layout, $this->GetRequestRecordID(), $this->sendData, $this->sendOptions);

      if (! $this->fm->getIsError($apiResult)) {

         // FX with the XML interface returns the record data and the meta data.
         // We need to make an additional calls to get the edited record and the meta data.
         $recordID = $this->GetRequestRecordID();
         $result = $this->TranslateResponse($this->fm->apiGetRecord($this->FX->layout, $recordID), true, $recordID);

      }
      else {
         $errorInfo = $this->fm->getMessageInfo($apiResult);

         $theError = $errorInfo[0];
         $result = new FX_Error($theError[FM_MESSAGE], $theError[FM_CODE]);
         $this->FX->fxError = $theError[FM_CODE];

         fmLogger('Found error(s):');
         fmLogger($errorInfo);
      }

      return $result;
   }

   // *********************************************************************************************************************************
   function FMDelete()
   {
      $result = false;

      $this->CreatePayload($this->action);

      $apiResult = $this->fm->apiDeleteRecord($this->FX->layout, $this->GetRequestRecordID(), $this->sendOptions);

      if (! $this->fm->getIsError($apiResult)) {
         $result = true;
         $this->FX->fxError = 0;
      }
      else {
         $errorInfo = $this->fm->getMessageInfo($apiResult);

         $theError = $errorInfo[0];
         $result = new FX_Error($theError[FM_MESSAGE], $theError[FM_CODE]);
         $this->FX->fxError = $theError[FM_CODE];

         fmLogger('Found error(s):');
         fmLogger($errorInfo);
      }

      return $result;
   }

   // *********************************************************************************************************************************
   function FMDuplicate()
   {
      $result = false;

      $this->CreatePayload($this->action);

      $apiResult = $this->fm->apiDuplicateRecord($this->FX->layout, $this->GetRequestRecordID(), $this->sendOptions);

      if (! $this->fm->getIsError($apiResult)) {
         $response = $this->fm->getResponse($apiResult);
         $recordID = $response[FM_RECORD_ID];              // Use recordID of duplicated record, not original

         // FX with the XML interface returns the record data and the meta data.
         // We need to make an additional calls to get the duplicated record and the meta data.
         $result = $this->TranslateResponse($this->fm->apiGetRecord($this->FX->layout, $recordID), true, $recordID);

      }
      else {
         $errorInfo = $this->fm->getMessageInfo($apiResult);

         $theError = $errorInfo[0];
         $result = new FX_Error($theError[FM_MESSAGE], $theError[FM_CODE]);
         $this->FX->fxError = $theError[FM_CODE];

         if ($theError[FM_CODE] == 1630) {                                                   // FileMaker Server 17 will return 1630
            fmLogger('Warning: FileMaker Server 18 required to duplicate a record.');
         }
         else {
            fmLogger('Found error(s):');
            fmLogger($errorInfo);
         }
      }

      return $result;
   }

   // *********************************************************************************************************************************
   function FMFind()
   {
      $result = false;

      $this->CreatePayload($this->action);

      $result = $this->TranslateResponse($this->fm->apiFindRecords($this->FX->layout, $this->sendData, $this->sendOptions), true);

      return $result;
   }

   // *********************************************************************************************************************************
   function FMFindAll($limit = '')
   {
      $this->CreatePayload($this->action);

      if ($limit != '') {
         $this->sendOptions['limit'] = $limit;
      }

      $result = $this->TranslateResponse($this->fm->apiGetRecords($this->FX->layout, $this->sendData, $this->sendOptions), true);

      return $result;
   }

   // *********************************************************************************************************************************
   function FMMetaData($recordID = '')
   {
      $result = true;

      $recordID = ($recordID != '') ? $recordID : $this->GetRequestRecordID();

      $apiResult = $this->fm->apiLayoutMetadata($this->FX->layout, $recordID);

      if (! $this->fm->getIsError($apiResult)) {
         $this->FX->fxError = 0;
         $response = $this->fm->getResponse($apiResult);


         // Map the fields definitions
         if (array_key_exists('fieldMetaData', $response)) {
            foreach ($response['fieldMetaData'] as $fieldMetaData) {
               $metaData = array();

               $metaData['emptyok'] = (array_key_exists('notEmpty', $fieldMetaData)) ? (($fieldMetaData['notEmpty'] == 'true') ? 'no' : 'yes'): '';
               $metaData['maxrepeat'] = (array_key_exists('maxRepeat', $fieldMetaData)) ? $fieldMetaData['maxRepeat'] : '';
               $metaData['name'] = (array_key_exists('name', $fieldMetaData)) ? $fieldMetaData['name'] : '';
               $metaData['extra'] = '';                                       // for compatibility w/ SQL databases
               $metaData['type'] = (array_key_exists('displayType', $fieldMetaData)) ? strtoupper($fieldMetaData['displayType']) : '';
               $metaData['valuelist'] = (array_key_exists('valueList', $fieldMetaData)) ? $fieldMetaData['valueList'] : '';

               $this->FX->fieldInfo[] = $metaData;
            }
         }


         // Map the portal field definitions
         if (array_key_exists('portalMetaData', $response)) {
            foreach ($response['portalMetaData'] as $portalMetaData) {
               foreach ($portalMetaData as $portalField) {
                  $metaData = array();
                  $metaData['name'] = (array_key_exists('name', $portalField)) ? $portalField['name'] : '';
                  $metaData['extra'] = '';                                       // for compatibility w/ SQL databases
                  $metaData['type'] = (array_key_exists('displayType', $portalField)) ? strtoupper($portalField['displayType']) : '';
                  $metaData['valuelist'] = (array_key_exists('valueList', $portalField)) ? $portalField['valueList'] : '';

                  $this->FX->fieldInfo[] = $metaData;
               }
            }
         }

         // Map the value lists
         if (array_key_exists('valueLists', $response)) {

            $valueLists = array();
            foreach ($response['valueLists'] as $valueList) {
               $values = array();
               foreach ($valueList['values'] as $value) {
                  $values[] = $value['value'];
               }
               $valueLists[$valueList['name']] = $values;

            }
            $this->FX->valueLists = $valueLists;
         }
      }
      else {
         $errorInfo = $this->fm->getMessageInfo($apiResult);

         $theError = $errorInfo[0];
         $result = new FX_Error($theError[FM_MESSAGE], $theError[FM_CODE]);
         $this->FX->fxError = $theError[FM_CODE];

         if ($theError[FM_CODE] == 1630) {                                                   // FileMaker Server 17 will return 1630
            fmLogger('Warning: FileMaker Server 18 required for meta data retrieval.');
         }
         else {
            fmLogger('Found error(s):');
            fmLogger($errorInfo);
         }
      }

      return $result;
   }

   // *********************************************************************************************************************************
   function FMDBNames()
   {
      $result = true;

      $apiResult = $this->fm->apiListDatabases();

      if (! $this->fm->getIsError($apiResult)) {
         $this->FX->fxError = 0;
         $response = $this->fm->getResponse($apiResult);

         if (array_key_exists('databases', $response)) {
            $index = 0;
            foreach ($response['databases'] as $database) {
               $this->FX->currentData['0' .'.'. $index] = array('DATABASE_NAME' => array($database['name']));
               $index++;
            }
         }
      }
      else {
         $errorInfo = $this->fm->getMessageInfo($apiResult);

         $theError = $errorInfo[0];
         $result = new FX_Error($theError[FM_MESSAGE], $theError[FM_CODE]);
         $this->FX->fxError = $theError[FM_CODE];

         if ($theError[FM_CODE] == 10) {                                                   // FileMaker Server 17 will return 10
            fmLogger('Warning: FileMaker Server 18 required for listing databases.');
         }
         else {
            fmLogger('Found error(s):');
            fmLogger($errorInfo);
         }
      }

      return $result;
   }

   // *********************************************************************************************************************************
   function FMLayoutNames()
   {
      $result = true;

      $apiResult = $this->fm->apiListLayouts();

      if (! $this->fm->getIsError($apiResult)) {
         $this->FX->fxError = 0;
         $response = $this->fm->getResponse($apiResult);

         if (array_key_exists('layouts', $response)) {
            // The Data API doesn't return the 'recordID' of the layout so we can't emulate
            // what FX does in this case. So we don't use any type of key on the array.
            foreach ($response['layouts'] as $layout) {
               $this->FX->currentData[] = array('LAYOUT_NAME' => array($layout['name']));
            }
         }
      }
      else {
         $errorInfo = $this->fm->getMessageInfo($apiResult);

         $theError = $errorInfo[0];
         $result = new FX_Error($theError[FM_MESSAGE], $theError[FM_CODE]);
         $this->FX->fxError = $theError[FM_CODE];

         if ($theError[FM_CODE] == 1630) {                                                   // FileMaker Server 17 will return 1630
            fmLogger('Warning: FileMaker Server 18 required for listing layouts.');
         }
         else {
            fmLogger('Found error(s):');
            fmLogger($errorInfo);
         }
      }

      return $result;
   }

   // *********************************************************************************************************************************
   function FMScriptNames()
   {
      $result = true;

      $apiResult = $this->fm->apiListScripts();

      if (! $this->fm->getIsError($apiResult)) {
         $this->FX->fxError = 0;
         $response = $this->fm->getResponse($apiResult);

         if (array_key_exists('scripts', $response)) {
            // The Data API doesn't return the 'recordID' of the layout so we can't emulate
            // what FX does in this case. So we don't use any type of key on the array.
            foreach ($response['scripts'] as $script) {
               $this->FX->currentData[] = array('SCRIPT_NAME' => array($script['name']));
            }
         }
      }
      else {
         $errorInfo = $this->fm->getMessageInfo($apiResult);

         $theError = $errorInfo[0];
         $result = new FX_Error($theError[FM_MESSAGE], $theError[FM_CODE]);
         $this->FX->fxError = $theError[FM_CODE];

         if ($theError[FM_CODE] == 1630) {                                                   // FileMaker Server 17 will return 1630
            fmLogger('Warning: FileMaker Server 18 required for listing scripts.');
         }
         else {
            fmLogger('Found error(s):');
            fmLogger($errorInfo);
         }
      }

      return $result;
   }


}

?>